<?php

namespace Give\Donors\Endpoints;

use Give\Donors\ListTable\DonorsListTable;
use Give\Donations\ValueObjects\DonationMetaKeys;
use Give\Framework\Database\DB;
use Give\Framework\QueryBuilder\QueryBuilder;
use WP_REST_Request;
use WP_REST_Response;

class ListDonors extends Endpoint
{
    /**
     * @var string
     */
    protected $endpoint = 'admin/donors';

    /**
     * @var WP_REST_Request
     */
    protected $request;

    /**
     * @var DonorsListTable
     */
    protected $listTable;

    /**
     * @inheritDoc
     */
    public function registerRoute()
    {
        register_rest_route(
            'give-api/v2',
            $this->endpoint,
            [
                [
                    'methods' => 'GET',
                    'callback' => [$this, 'handleRequest'],
                    'permission_callback' => [$this, 'permissionsCheck'],
                ],
                'args' => [
                    'page' => [
                        'type' => 'integer',
                        'required' => false,
                        'default' => 1,
                        'minimum' => 1
                    ],
                    'perPage' => [
                        'type' => 'integer',
                        'required' => false,
                        'default' => 30,
                        'minimum' => 1
                    ],
                    'donations' => [
                        'type' => 'integer',
                        'required' => false,
                        'default' => 0
                    ],
                    'search' => [
                        'type' => 'string',
                        'required' => false
                    ],
                    'start' => [
                        'type' => 'string',
                        'required' => false,
                        'validate_callback' => [$this, 'validateDate']
                    ],
                    'campaignId' => [
                        'type' => 'integer',
                        'required' => false,
                        'default' => 0
                    ],
                    'end' => [
                        'type' => 'string',
                        'required' => false,
                        'validate_callback' => [$this, 'validateDate']
                    ],
                    'sortColumn' => [
                        'type' => 'string',
                        'required' => false,
                        'sanitize_callback' => 'sanitize_text_field',
                    ],
                    'sortDirection' => [
                        'type' => 'string',
                        'required' => false,
                        'enum' => [
                            'asc',
                            'desc'
                        ],
                    ],
                    'locale' => [
                        'type' => 'string',
                        'required' => false,
                        'default' => get_locale(),
                    ],
                    'return' => [
                        'type' => 'string',
                        'required' => false,
                        'default' => 'columns',
                        'enum' => [
                            'model',
                            'columns'
                        ],
                    ],
                ],
            ]
        );
    }

    /**
     * @since 2.24.0 Change this to use the new ListTable class
     * @since 2.20.0
     *
     * @param WP_REST_Request $request
     *
     * @return WP_REST_Response
     */
    public function handleRequest(WP_REST_Request $request): WP_REST_Response
    {
        $this->request = $request;
        $this->listTable = give(DonorsListTable::class);

        $donors = $this->getDonors();
        $donorsCount = $this->getTotalDonorsCount();
        $pageCount = (int)ceil($donorsCount / $request->get_param('perPage'));

        if ('model' === $this->request->get_param('return')) {
            $items = $donors;
        } else {
            $this->listTable->items($donors, $this->request->get_param('locale') ?? '');
            $items = $this->listTable->getItems();
        }

        return new WP_REST_Response(
            [
                'items' => $items,
                'totalItems' => $donorsCount,
                'totalPages' => $pageCount
            ]
        );
    }

    /**
     * @since 2.24.0 Replace Query Builder with Donors model
     * @since 2.21.0
     *
     * @return array
     */
    public function getDonors(): array
    {
        $page = $this->request->get_param('page');
        $perPage = $this->request->get_param('perPage');
        $sortColumns = $this->listTable->getSortColumnById($this->request->get_param('sortColumn') ?: 'id');
        $sortDirection = $this->request->get_param('sortDirection') ?: 'desc';

        $query = give()->donors->prepareQuery();
        $query = $this->getWhereConditions($query);

        foreach ($sortColumns as $sortColumn) {
            $query->orderBy($sortColumn, $sortDirection);
        }

        $query->limit($perPage)
            ->offset(($page - 1) * $perPage);

        $donors = $query->getAll();

        if (!$donors) {
            return [];
        }

        return $donors;
    }

    /**
     * @since 2.24.0 Replace Query Builder with Donors model
     * @since 2.21.0
     *
     * @return int
     */
    public function getTotalDonorsCount(): int
    {
        $query = DB::table('give_donors');
        $query = $this->getWhereConditions($query);

        return $query->count();
    }

    /**
     * @since 2.24.0 Replace Query Builder with Donors model
     * @since 2.21.0
     *
     * @param QueryBuilder $query
     *
     * @return QueryBuilder
     */
    private function getWhereConditions(QueryBuilder $query): QueryBuilder
    {
        $search = $this->request->get_param('search');
        $start = $this->request->get_param('start');
        $end = $this->request->get_param('end');
        $campaignId = $this->request->get_param('campaignId');

        if ($search) {
            if (ctype_digit($search)) {
                $query->where('id', $search);
            } else {
                $query->whereLike('name', $search);
                $query->orWhereLike('email', $search);
            }
        }

        if ($start && $end) {
            $query->whereBetween('date_created', $start, $end);
        } else if ($start) {
            $query->where('date_created', $start, '>=');
        } else if ($end) {
            $query->where('date_created', $end, '<=');
        }

        if ($campaignId) {
            $query
                ->whereIn('id', static function (QueryBuilder $builder) use ($campaignId) {
                    $builder
                        ->from('give_donationmeta')
                        ->distinct()
                        ->select('meta_value')
                        ->where('meta_key', '_give_payment_donor_id')
                        ->whereIn('donation_id', static function (QueryBuilder $builder) use ($campaignId) {
                            $builder
                                ->from('give_donationmeta')
                                ->select('donation_id')
                                ->where('meta_key', DonationMetaKeys::CAMPAIGN_ID)
                                ->where('meta_value', $campaignId);
                        });
                });
        }

        return $query;
    }
}
